/*
*  Arnold emulator (c) Copyright, Kevin Thacker 1995-2015
*
*  This file is part of the Arnold emulator source code distribution.
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
/* Hackit - good */

#include "cpc.h"
#include "emudevice.h"
#include "riff.h"

typedef struct
{
	int ROMLatch;
	BOOL SwitchEnable;
	unsigned char SystemROM[16384];
} HackIt;

static HackIt hack;

void hackit_SetROM( const unsigned char *pROM, unsigned long RomLength )
{
	EmuDevice_CopyRomData( hack.SystemROM, sizeof( hack.SystemROM ), pROM, RomLength );
}

void hackit_ClearROM(void)
{
	EmuDevice_ClearRomData(hack.SystemROM, sizeof(hack.SystemROM));
}


static EmuDeviceRom HackitRom[1] =
{
	{
		"HackIt System ROM",
		"SystemRom",
		hackit_SetROM,
		hackit_ClearROM,
		false,
		false,
		16384,
		0   /* ROM CRC - todo */
	}
};

void    hackit_SetSwitch( BOOL bState )
{
	hack.SwitchEnable = bState;
}

BOOL    hackit_IsSwitchEnabled( void )
{
	return hack.SwitchEnable;
}

void	hackit_ROMSelect( Z80_WORD Port, Z80_BYTE Data )
{
	/* d0,d1,d2 goes into A,B,C of 74LS138 */
	/* d3 goes into G2A which is active low */
	hack.ROMLatch = Data & 0x0f;

	Computer_RethinkMemory();

}

CPCPortWrite hackitPortWrites[ 1 ] =
{
	{
		0x02000,            /* and */
		0x00000,            /* compare */
		hackit_ROMSelect
	}
};

void hackit_MemoryRethinkHandler( MemoryData *pData )
{
	/* uses ROMEN */
	if ( pData->bRomEnable[ 6 ] && hack.SwitchEnable && (hack.ROMLatch==0x0) )
	{
		/* disable other roms in the chain */
		pData->bRomDisable[ 6 ] = TRUE;
		pData->bRomDisable[ 7 ] = TRUE;

		/* Hackit ROM */
		unsigned char *pRomPtr = hack.SystemROM - 0x0c000;
		pData->pReadPtr[ 7 ] = pRomPtr;
		pData->pReadPtr[ 6 ] = pRomPtr;
		pData->pReadMaskPtr[7] = GetDefaultReadMask() - 0x0c000;
		pData->pReadMaskPtr[6] = GetDefaultReadMask() - 0x0c000;

	}
}

/* two switches, one of which is a busreset */
static EmuDeviceSwitch hackitSwitches[ 1 ] =
{
	{
		"Activate Switch",          /* the switch on the top */
		"Activate",
		hackit_IsSwitchEnabled,
		hackit_SetSwitch
	},
};

void hackitDevice_Init()
{
	hack.ROMLatch = 0;
	hack.SwitchEnable = FALSE;
}

static EmuDevice hackitDevice =
{
	NULL,
	hackitDevice_Init,
	NULL,
	"hackit",
	"Hackit",
	"HackIt/Le Hacker",
	CONNECTION_EXPANSION,   /* connected to expansion */
	DEVICE_FLAGS_HAS_EXPANSION_ROMS | DEVICE_FLAGS_HAS_PASSTHROUGH| DEVICE_WORKING,
	0,                /* no read ports */
	NULL,
	sizeof(hackitPortWrites)/sizeof(hackitPortWrites[0]),                    /* 2 write ports */
	hackitPortWrites, /* the ports */
	0,
	NULL,
	0,
	NULL,
	NULL, /* reset function */
	hackit_MemoryRethinkHandler, /* memory rethink */
	NULL, /* power function */
	sizeof( hackitSwitches ) / sizeof( hackitSwitches[ 0 ] ),                      /* 3 switch */
	hackitSwitches,
	0,                      /* has 1 button which is reset; not currently emulated */
	NULL,
	0,
	NULL,
	sizeof(HackitRom)/sizeof(HackitRom[0]),                      /* no onboard roms */
	HackitRom,
	NULL,	/* no cursor update function */
	NULL,
	NULL,	/* printer */
	NULL,	/* joystick */
	0,
	NULL,
	NULL,	/* no sound */
	NULL, /* lpen */
	NULL, /* reti */
	NULL, /* ack maskable interrupt */
	NULL, /* dkram data */
	NULL, /* device ram */
	NULL, /* device backup */
	NULL,
};

void hackit_Init()
{
	RegisterDevice( &hackitDevice );
}









